/*:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
::
::   Module      :   CJ125 LSU Resource Framework API Header File
::   Copyright   :   (C)2003-2009 Woodward
::   Platform(s) :   MPC5xx
::   Limitations :   Dependant upon MPC5xx OS sub-project
::
:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::*/
/*! \file Resource_CJ125_LSU.h
    \brief This behaviour abstracts the functionality of the Bosch CJ125 integrated circuit that is employed
           on some Woodward modules. The CJ125 integrated circuit is a control and amplifier circuit for a
           wide range lambda Sensor LSU4.x for the continuous regulation of lambda in combination with the
           sensor in the range of lambda = 0.65... infinite (air).
           
The behaviour provides a mechanism through which the internal registers of the integrated circuit can be
read and written by the application. The ancillary resources that are linked to the integrated circuit like
the heater output control and the analogue inputs associated with the sensor have also been bundled into the
CJ125 LSU object and are accessible through this behaviour.
- \ref cj125create
- \ref cj125read
- \ref cj125setattrib
*/

#ifndef __RESOURCE_CJ125_LSU_H
#define __RESOURCE_CJ125_LSU_H

/*----- INCLUDES ------------------------------------------------------------------------------------------*/
#include <typedefn.h>
#include <resource.h>

/*----- DEFINES -------------------------------------------------------------------------------------------*/
#define USE_ALL_CJ125_WRITE_REGS     (USE_CJ125_INIT1_REG | USE_CJ125_INIT2_REG)

/*----- TYPEDEFS ------------------------------------------------------------------------------------------*/
#pragma pack(1)
/*! \brief Enumeration describes the set of mask definitions that are used to identify the valid CJ125
           attributes in the S_CJ125_LSU_WriteAttributes, S_CJ125_LSUCreateAttributes and S_CJ125_LSU_ReadAttributes
           data structures.

Each mask definition applies to a particular member of one of the data structures.
\code
    ...
    S_CJ125_LSU_WriteAttributes AttribsObj;

    // Going to set the INT1 and INIT2 CJ125 registers so include in the mask
    AttribsObj.uValidAttributesMask = USE_CJ125_LSU_INIT1_REG | USE_CJ125_LSU_INIT2_REG;
    // The remainder of the data structure can be left undefined since the uValidAttributesMask
    //   indicates that the members are not valid
    ... \endcode */
typedef enum
{
/*! Selects S_CJ125_LSU_WriteAttributes::u4Frequency */
    USE_CJ125_LSU_PWM_FREQUENCY     = 0x0001,   /* DO NOT CHANGE - Linked to USE_PWM_FREQUENCY */
/*! Selects S_CJ125_LSU_WriteAttributes::s2DutyCycle */
    USE_CJ125_LSU_PWM_DUTYCYCLE     = 0x0002,   /* DO NOT CHANGE - Linked to USE_PWM_DUTYCYCLE */
/*! Selects S_CJ125_LSU_WriteAttributes::eResourceCondition */
    USE_CJ125_LSU_CONDITION         = 0x0004,   /* DO NOT CHANGE - Linked to USE_CONDITION */
    USE_CJ125_PWM_DYNAMIC_RESERVED1 = 0x0008,   /* DO NOT CHANGE - Linked to USE_PWM_DIAG_COND */
    USE_CJ125_PWM_DYNAMIC_RESERVED2 = 0x0010,   /* DO NOT CHANGE - Linked to USE_PWM_DYNAMIC_RESERVED2 */
    USE_CJ125_PWM_DYNAMIC_RESERVED3 = 0x0020,   /* DO NOT CHANGE - Linked to USE_PWM_DYNAMIC_RESERVED3 */
/*! Selects S_CJ125_LSU_ReadAttributes::DIAGRegObj */
    USE_CJ125_LSU_DIAG_REG          = 0x0040,
/*! Selects S_CJ125_LSU_ReadAttributes::INIT1RegObj and S_CJ125_LSU_WriteAttributes::INIT1RegObj */
    USE_CJ125_LSU_INIT1_REG         = 0x0080,
/*! Selects S_CJ125_LSU_ReadAttributes::INIT2RegObj and S_CJ125_LSU_WriteAttributes::INIT2RegObj */
    USE_CJ125_LSU_INIT2_REG         = 0x0100,
/*! Selects S_CJ125_LSU_ReadAttributes::IDENTRegObj */
    USE_CJ125_LSU_IDENT_REG         = 0x0200,
/*! Selects S_CJ125_LSU_ReadAttributes::u2ADC_UA */
    USE_CJ125_LSU_UA_ADC_VALUE      = 0x0400,
/*! Selects S_CJ125_LSU_ReadAttributes::u2ADC_UR */
    USE_CJ125_LSU_UR_ADC_VALUE      = 0x0800,
/*! Selects S_CJ125_LSUCreateAttributes::DynamicObj */
    USE_CJ125_LSU_DYNAMIC_ON_CREATE = 0x1000,

/* IF THIS TYPE EXCEEDS 0x8000 THEN ALTER THE TYPE OF THE uValidAttributesMask MEMBER(S) ACCORDINGLY */

} E_CJ125AttributesMask;

/* :::::::::::::::::::::::::::::::
   :: DIAG Register
   :::::::::::::::::::::::::::::*/
/*! CJ125 Diagnostic information. See the CJ125 data sheet for furter information */
/* Name: E_CJ125DiagInfo ClassID:ENUMDEC EnumDflt:"Illegal"        */
typedef enum
{
/*! Shorted to Ground */
    CJ125_SHORT_TO_GND = 0,        /* EnumTxt:"Shorted to Ground"  */
/*! Low voltage */
    CJ125_LOW_VOLTAGE = 1,         /* EnumTxt:"Low Voltage"        */
/*! Short circuit to battery */
    CJ125_SHORT_TO_VBATT = 2,      /* EnumTxt:"Shorted To Battery" */
/*! No fault or diagnostic disabled */
    CJ125_NO_FAULT = 3             /* EnumTxt:"Operating Normally" */
} E_CJ125DiagInfo;

/*! CJ125 Diagnostic information. See the CJ125 data sheet for furter information */
/* Name: E_CJ125HeaterDiagInfo ClassID:ENUMDEC EnumDflt:"Illegal"  */
typedef enum
{
/*! The heater load is shorted to ground */
    CJ125_HEAT_SHORT_TO_GND = 0,   /* EnumTxt:"Heater Shorted to Ground"  */
/*! The heater load is open circuit */
    CJ125_HEAT_OPEN_LOAD = 1,      /* EnumTxt:"Heater Load Open"          */
/*! The heater load is shorted to battery */
    CJ125_HEAT_SHORT_TO_VBATT = 2, /* EnumTxt:"Heater Shorted To Battery" */
/*! The heater load is operating normally, which is the RESET value */
    CJ125_HEAT_NO_FAULT = 3        /* EnumTxt:"Operating Normally" */
} E_CJ125HeaterDiagInfo;

/*! \brief The Bosch CJ125 \b DIAG_REG register, detailed in the CJ125 data sheet.

    The value of this register can be read via the GetResourceAttributes_BEHAVIOUR_CJ125_LSU() function. See \ref cj125regread */
typedef struct
{
/*! \brief Heater Diagnostic */
    E_CJ125HeaterDiagInfo eHeaterDiag : 2;
/*! \brief IA and IP diagnostic */
    E_CJ125DiagInfo eIADiag : 2;
/*! \brief UN input diagnostic */
    E_CJ125DiagInfo eUNDiag : 2;
/*! \brief VM input diagnostic */
    E_CJ125DiagInfo eVMDiag : 2;
} S_CJ125Diag;

/* :::::::::::::::::::::::::::::::
   :: IDENT Register
   :::::::::::::::::::::::::::::*/
/*! \brief The Bosch CJ125 \b IDENT_REG register, detailed in the CJ125 data sheet.

    The value of this register is fixed at manufacture. It can be read as a single object via the
    GetResourceAttributes_BEHAVIOUR_CJ125_LSU() function. See \ref cj125regread */
typedef struct
{
    uint1 u1ICNumber     : 5;   /*!< The IC number */
    uint1 u1MaskVersion  : 3;   /*!< The mask number */
} S_CJ125Ident;

/*! \brief The Bosch CJ125 \b IDENT_REG register.

    The value of this register is fixed at manufacture. It can be read as a single object via the
    GetResourceAttributes_BEHAVIOUR_CJ125_LSU() function. See \ref cj125regread */
typedef union
{
/*! The Bosch CJ125 \b IDENT_REG as detailed in the CJ125 data sheet. */
    S_CJ125Ident VersionObj;
/*! The INDENT_REG treated as a single number */
    uint1 u1Version;
} U_CJ125Ident;

/* :::::::::::::::::::::::::::::::
   :: INIT1 Register
   :::::::::::::::::::::::::::::*/
/*! Enumeration describing the valid values for the \b EN_HOLD field of the CJ125's \b INIT1_REG */
typedef enum
{
    CJ125_DISABLE_HOLD, /*!< Pump current control without HOLD-phase. Not a recommended setting */
    CJ125_ENABLE_HOLD,  /*!< Enable hold used for Measurement mode, calibration mode */
} E_CJ125_HOLD;

/*! Enumeration describing the valid values of the \b PA field of the CJ125's \b INIT1_REG */
typedef enum
{
    CJ125_PA_ON,    /*!< ON, Pump control current released */
    CJ125_PA_OFF,   /*!< OFF, IA high impedance, diagnostic at IA off: write access in \b DIAG_REG for DIA4 and DIA5 disabled. */
} E_CJ125_PA;

/*! Enumeration describing the valid values of the \b RA field of the CJ125's \b INIT1_REG */
typedef enum
{
    CJ125_RA_MEASURE,   /*!< Measurement mode */
    CJ125_RA_CALIBRATE  /*!< Calibration mode */
} E_CJ125_RA;

/*! Enumeration describing the valid values of the \b LA field of the CJ125's \b INIT1_REG */
typedef enum
{
    CJ125_LA_MEASURE,   /*!< Measurement mode for the lambda signal at [UA] */
    CJ125_LA_ADJUST     /*!< Adjustment mode for the lambda signal at [UA] */
} E_CJ125_LA;

/*! Enumeration describing the valid values of the \b VL field of the CJ125's \b INIT1_REG */
typedef enum
{
    CJ125_AMP_FOR_LAMBDA_065,   /*!< Amplification set to 8, which gives lambda range between 0.65 and infinite */
    CJ125_AMP_FOR_LAMBDA_075    /*!< Amplification set to 11, which gives lambda range between 0.75 and infinite */
} E_CJ125_VL;
 
/*! \brief The Bosch CJ125 \b INIT_REG1, detailed in the CJ125 data sheet.

    This object can be written via the SetResourceAttributes_BEHAVIOUR_CJ125_LSU() function and read via the
    GetResourceAttributes_BEHAVIOUR_CJ125_LSU() function. See \ref cj125regread and \ref cj125regwrites */
typedef struct
{
    E_CJ125_HOLD eHOLD  : 1;    /*!< \b EN_HOLD */
    E_CJ125_PA ePA      : 1;    /*!< \b PA */
    bool1 b1Unused2     : 1;
    E_CJ125_RA  eRA     : 1;    /*!< \b RA */
    bool1 b1EN_F3K      : 1;    /*!< \b EN_F3K, FALSE sets F3K OFF, TRUE sets F3K ON */
    E_CJ125_LA eLA      : 1;    /*!< \b LA */
    bool1 b1Unused1     : 1;
    E_CJ125_VL eVL      : 1;    /*!< \b VL */
} S_CJ125Init1;

/*! \brief The Bosch CJ125 \b INIT_REG2, detailed in the CJ125 data sheet.

    This object can be written via the SetResourceAttributes_BEHAVIOUR_CJ125_LSU() function and read via the
    GetResourceAttributes_BEHAVIOUR_CJ125_LSU() function. See \ref cj125regread and \ref cj125regwrites */
typedef struct
{
    bool1 b1Unused1   : 1;
    bool1 b1SRESET    : 1;  /*!< Software reset of SPI and all the CJ125's registers when TRUE. */
    bool1 b1SET_DIA_Q : 1;  /*!< - FALSE Short circuit detection to GND at IA/IP for VUP > VUN ("lean") and voltage below threshold
                                 - TRUE short circuit dedection to GND at IA/IP when voltage below threshold */
    bool1 b1ENSCUN    : 1;  /*!< - FALSE short circuit detection to Vbat at UN disabled. Used when sensor is high impedance and
                                   pump reference currents are turned on.
                                 - TRUE short circuit detection to Vbat at UN enabled */
    bool1 b1_80uAOn   : 1;  /*!< Boolean indicating whether pump reference current 80uA is turned ON */
    bool1 b1_40uAOn   : 1;  /*!< Boolean indicating whether pump reference current 40uA is turned ON */
    bool1 b1_20uAOn   : 1;  /*!< Boolean indicating whether pump reference current 20uA is turned ON */
    bool1 b1_10uAOn   : 1;  /*!< Boolean indicating whether pump reference current 10uA is turned ON */
} S_CJ125Init2;

/*! \brief This data structure describes all of the CJ125 resource's readable attributes.

    The attributes are read through the use of GetResourceAttributesBEHAVIOUR_CJ125_LSU(). The data
    structure does not need to be completely filled by the framework if only certain attributes are of interest.
    The \b uValidAttributesMask is a bit field member that is used to identify which attributes should be read.
    Each attribute is identified with a separate bit mask that is logic-ORed into the mask when it's to be used.
    \code
    ...
    S_CJ125_LSU_ReadAttributes ReadObj;

    // Only want the two ADC values
    ReadObj.uValidAttributesMask = USE_CJ125_LSU_UA_ADC_VALUE | USE_CJ125_LSU_UR_ADC_VALUE; \endcode */
typedef struct
{
/*! Logic-OR the attributes [\ref USE_CJ125_LSU_UA_ADC_VALUE, \ref USE_CJ125_LSU_UR_ADC_VALUE, \ref USE_CJ125_LSU_DIAG_REG,
    \ref USE_CJ125_LSU_IDENT_REG, \ref USE_CJ125_LSU_INIT1_REG and \ref USE_CJ125_LSU_INIT2_REG ] that are valid for
    this instance of the data structure.

    \code
    ...
    S_CJ125_LSU_ReadAttributes ReadObj;

    // Only want the two ADC values
    ReadObj.uValidAttributesMask = USE_CJ125_LSU_UA_ADC_VALUE | USE_CJ125_LSU_UR_ADC_VALUE;
    // The rest of the data structure will remain unmodified upon return from GetResourceAttributesBEHAVIOUR_CJ125_LSU() \endcode */    
    uint2 uValidAttributesMask;
    S_CJ125Diag  DIAGRegObj;    /*!< Value read from the CJ125 diagnostic register. */
    U_CJ125Ident IDENTRegObj;   /*!< Value read from the CJ125 Identification register */
    S_CJ125Init1 INIT1RegObj;   /*!< Current value of the configuration information within the CJ125 integrated circuit. */
    S_CJ125Init2 INIT2RegObj;   /*!< Current value of the configuration information within the CJ125 integrated circuit. */
    uint2 u2ADC_UA;             /*!< UA ADC value. */
    uint2 u2ADC_UR;             /*!< UR ADC value. */
} S_CJ125_LSU_ReadAttributes;

typedef S_CJ125_LSU_ReadAttributes* S_CJ125_LSU_ReadAttributesPtr;

/*! \brief This data structure describes all of the CJ125 resource's runtime configuration attributes.

    The attributes are altered through the use of SetResourceAttributesBEHAVIOUR_CJ125_LSU(). The data
    structure does not need to be completely filled inorder to be used. The \b uValidAttributesMask is a bit
    field member that is used to identify which attributes are valid. Each attribute is identified with a
    separate bit mask that is logic-ORed into the mask when it's to be used.
    \code
    ...
    S_CJ125_LSU_WriteAttributes WriteObj;

    // Only going to set the Duty Cycle, and Frequency members
    WriteObj.uValidAttributesMask = USE_CJ125_LSU_PWM_FREQUENCY | USE_CJ125_LSU_PWM_DUTYCYCLE; \endcode */
typedef struct
{
/*! Logic-OR the attributes [\ref USE_CJ125_LSU_PWM_FREQUENCY, \ref USE_CJ125_LSU_PWM_DUTYCYCLE, \ref USE_CJ125_LSU_CONDITION,
    \ref USE_CJ125_LSU_INIT1_REG and \ref USE_CJ125_LSU_INIT2_REG ] that are valid for this this instance of the data structure.

    \code
    ...
    S_CJ125_LSU_WriteAttributes WriteObj;

    // Only going to set the Duty Cycle, and Frequency members
    WriteObj.uValidAttributesMask = USE_CJ125_LSU_PWM_FREQUENCY | USE_CJ125_LSU_PWM_DUTYCYCLE;
    WriteObj.s2PWMDutyCycle = (10 * FULL_SCALE_DUTYCYCLE) / 100L;
    WriteObj.u4PWMFrequency = 1000 * FREQUENCY_API_SCALING;
    // The rest of the data structure can remain uninitialised because the framework will ignore it \endcode */    
    uint2 uValidAttributesMask;
/*! The condition of the resource. A disabled resource remains in the safest state. Thus the heater output is OFF,
    regardless of what the other runtime attributes have been set to. Use the \ref USE_CJ125_LSU_CONDITION mask to
    select this attribute. The value of this attribute upon resource creation is \ref RES_DISABLED. */
    E_ResourceCond eResourceCondition;
/*! The INIT1 register. See the Bosch CJ125 data sheet for a detailed description on the members of this register.
    Select this attribute with the \ref USE_CJ125_LSU_INIT1_REG bit mask. The value of this attribute upon resource
    creation is its reset value - as defined by the Bosch CJ125 data sheet */
    S_CJ125Init1 INIT1RegObj;
/*! The INIT2 register. See the Bosch CJ125 data sheet for a detailed description on the members of this register.
    Select this attribute with the \ref USE_CJ125_LSU_INIT2_REG bit mask. The value of this attribute upon resource
    creation is its reset value - as defined by the Bosch CJ125 data sheet */
    S_CJ125Init2 INIT2RegObj;
/*! Defines the percentage of the pulse-width period that the heater output should be active or "ON". Has units of percentage
    where 100% is defined by \ref FULL_SCALE_DUTYCYCLE. Select this attribute with the \ref USE_CJ125_LSU_PWM_DUTYCYCLE bit mask.
    
    Also see \ref pwmdutyandfreq */
    sint2 s2PWMDutyCycle;
/*! Defines the frequency in Hertz, which are internally scaled by \ref FREQUENCY_API_SCALING. Thus FREQUENCY_API_SCALING==1Hz.
    Select this attribute with the \ref USE_CJ125_LSU_PWM_FREQUENCY bit mask.

    Also see \ref pwmdutyandfreq */
    uint4 u4PWMFrequency;
} S_CJ125_LSU_WriteAttributes;

typedef S_CJ125_LSU_WriteAttributes const* S_CJ125_LSU_WriteAttributesPtr;

/*! \brief This data structure describes the creation attributes for a CJ125 lamda sensing resource

    Included within the structure is the \p DynamicObj which represents the initial states for the resource's
    dynamic attributes. This data structure is referenced when the resource is created via the
    CreateResourceBEHAVIOUR_CJ125_LSU() function and \ref USE_CJ125_LSU_DYNAMIC_ON_CREATE is included in the
    S_CJ125_LSUCreateAttributes::uValidAttributesMask member.
    
    \code
    ...
    S_CJ125_LSUCreateAttributes CreateObj;

    // Will use DynamicObj
    CreateObj.uValidAttributesMask = USE_CJ125_LSU_DYNAMIC_ON_CREATE;
    // Only going to set the condition member
    CreateObj.DynamicObj.uValidAttributesMask = USE_CJ125_LSU_CONDITION;
    // The rest of the data structure can remain uninitialised because the framework will ignore it \endcode */    
typedef struct
{
/*! Logic-OR the attributes [\ref USE_CJ125_LSU_DYNAMIC_ON_CREATE] that are valid for this instance of the data structure.

    \code
    ...
    S_CJ125_LSUCreateAttributes CreateObj;

    // Will use DynamicObj
    CreateObj.uValidAttributesMask = USE_CJ125_LSU_DYNAMIC_ON_CREATE;
    // Only going to set the condition member
    CreateObj.DynamicObj.uValidAttributesMask = USE_CJ125_LSU_CONDITION;
    // The rest of the data structure can remain uninitialised because the framework will ignore it \endcode */    
    uint2 uValidAttributesMask;
/*! Initial values of the runtime attributes, which can be later altered through a call to
    SetResourceAttributesBEHAVIOUR_CJ125_LSU(). Select with the \ref USE_CJ125_LSU_DYNAMIC_ON_CREATE bit mask */
    S_CJ125_LSU_WriteAttributes DynamicObj;
} S_CJ125_LSUCreateAttributes;

typedef S_CJ125_LSUCreateAttributes const* S_CJ125_LSUCreateAttributesPtr;

#pragma pack()
/*----- EXTERNALS -----------------------------------------------------------------------------------------*/

/*----- PROTOTYPES ----------------------------------------------------------------------------------------*/
/* BEHAVIOUR_CJ125_LSU inherits from BEHAVIOUR_PWM so all its attribute access functions will work directly*/
/*   with this behaviour                                                                                   */

NativeError_S GetCJ125_LSU_DIAGRegisterValue(E_ModuleResource, S_CJ125Diag*);

NativeError_S CreateResourceBEHAVIOUR_CJ125_LSU(E_ModuleResource, S_CJ125_LSUCreateAttributes const*);
NativeError_S SetResourceAttributesBEHAVIOUR_CJ125_LSU(E_ModuleResource, S_CJ125_LSU_WriteAttributes const*);
NativeError_S GetResourceAttributesBEHAVIOUR_CJ125_LSU(E_ModuleResource, S_CJ125_LSU_ReadAttributes*);

#endif /* __RESOURCE_CJ125_LSU_H */

/*----- END OF FILE ---------------------------------------------------------------------------------------*/
